const TelegramBot = require('node-telegram-bot-api');
require('dotenv').config();

const token = process.env.TELEGRAM_BOT_TOKEN;
const bot = new TelegramBot(token, { polling: true });

const { Product, PurchaseLog } = require('../models');
const { createCheckoutSession } = require('../services/stripeService');

// 👋 START command
bot.onText(/\/start/, (msg) => {
  const chatId = msg.chat.id;

  const message =
    `👋 *Welcome to the Digital Store Bot!*\n\n` +
    `Use the following commands:\n` +
    `• /products — Browse available products\n` +
    `• /buy [id] — Purchase a product\n` +
    `• /myorders — View your past purchases\n` +
    `• /help — How to use the bot\n` +
    `• /contact — Contact support`;

  bot.sendMessage(chatId, message, {
    parse_mode: 'Markdown',
    reply_markup: {
      keyboard: [
        ['/products', '/myorders'],
        ['/help', '/contact']
      ],
      resize_keyboard: true,
      one_time_keyboard: false
    }
  });
});

// 🛍️ PRODUCTS command
bot.onText(/\/products/, async (msg) => {
  const chatId = msg.chat.id;

  try {
    const allProducts = await Product.findAll();
    const visibleProducts = [];

    for (const product of allProducts) {
      if (product.is_unlimited) {
        visibleProducts.push(product);
      } else {
        const available = await product.getAvailableStock?.();
        if (available > 0) {
          visibleProducts.push(product);
        }
      }
    }

    if (visibleProducts.length === 0) {
      return bot.sendMessage(chatId, '❌ No products available at the moment.');
    }

    let message = '🛍️ *Available Products:*\n\n';
    visibleProducts.forEach((p, index) => {
      message += `*${index + 1}. ${p.name}*\n`;
      message += `Type: \`${p.type}\`\n`;
      message += `Price: $${p.price}\n\n`;
    });

    message += '💡 Use the command /buy [product ID] to make a purchase.';

    bot.sendMessage(chatId, message, { parse_mode: 'Markdown' });
  } catch (err) {
    console.error(err);
    bot.sendMessage(chatId, '❌ Error loading products.');
  }
});

// 💳 BUY command
bot.onText(/\/buy (\d+)/, async (msg, match) => {
  const chatId = msg.chat.id;
  const productId = match[1];

  try {
    const product = await Product.findByPk(productId);

    if (!product) {
      return bot.sendMessage(chatId, '❌ Product not found.');
    }

    // Verificar stock antes de continuar
    if (!product.is_unlimited) {
      const stock = await product.getAvailableStock?.();
      if (!stock || stock === 0) {
        return bot.sendMessage(chatId, '⛔ This product is currently out of stock.');
      }

      if (product.is_reserved) {
        return bot.sendMessage(chatId, '⏳ This product is currently reserved by another user.');
      }

      // Marcar como reservado
      product.is_reserved = true;
      await product.save();
    } else {
      // Para productos ilimitados, verificar si ya fue usado (por si acaso)
      if (product.is_used) {
        return bot.sendMessage(chatId, '❌ This product has already been sold.');
      }
    }

    // Crear sesión de pago
    const url = await createCheckoutSession(product, chatId);
    bot.sendMessage(chatId, `💳 Click the link below to complete your payment:\n${url}`);
  } catch (err) {
    console.error(err);
    bot.sendMessage(chatId, '❌ Error creating payment link.');
  }
});

// 📈 SALES command
bot.onText(/\/sales (\d+)/, async (msg, match) => {
  const chatId = msg.chat.id;
  const productId = match[1];

  try {
    const product = await Product.findByPk(productId);

    if (!product) {
      return bot.sendMessage(chatId, '❌ Product not found.');
    }

    const salesCount = await PurchaseLog.count({
      where: { product_id: productId }
    });

    const message = 
      `📈 *Sales Report*\n\n` +
      `🛍️ *Product:* ${product.name}\n` +
      `🔢 *Total sales:* ${salesCount}`;

    bot.sendMessage(chatId, message, { parse_mode: 'Markdown' });
  } catch (err) {
    console.error(err);
    bot.sendMessage(chatId, '❌ Error retrieving sales data.');
  }
});

// 🧾 MYORDERS command
bot.onText(/\/myorders/, async (msg) => {
  const chatId = msg.chat.id;

  try {
    const orders = await PurchaseLog.findAll({
      where: { telegram_id: chatId.toString() },
      order: [['createdAt', 'DESC']],
      limit: 10
    });

    if (orders.length === 0) {
      return bot.sendMessage(chatId, '🧾 You have no previous orders.');
    }

    for (const o of orders) {
      const text =
        `🛍️ *${o.name}*\n` +
        `💼 Type: \`${o.type}\`\n` +
        `💵 Price: $${o.price}\n` +
        `📅 Date: ${new Date(o.createdAt).toLocaleDateString()}`;

      await bot.sendMessage(chatId, text, {
        parse_mode: 'Markdown',
        reply_markup: {
          inline_keyboard: [[
            { text: '📤 Resend Product', callback_data: `resend_${o.id}` }
          ]]
        }
      });
    }
  } catch (err) {
    console.error(err);
    bot.sendMessage(chatId, '❌ Error retrieving your orders.');
  }
});

bot.on('callback_query', async (query) => {
  const chatId = query.message.chat.id;
  const data = query.data;

  if (data.startsWith('resend_')) {
    const logId = data.split('_')[1];

    try {
      const log = await PurchaseLog.findByPk(logId);

      if (!log || log.telegram_id !== chatId.toString()) {
        return bot.answerCallbackQuery(query.id, { text: '❌ Order not found.', show_alert: true });
      }

      await bot.sendMessage(
        chatId,
        `📤 *Resent Product*\n\n` +
        `🛍️ *Product:* ${log.name}\n📦 *Type:* ${log.type}\n\n` +
        `🔑 *Key:* \`${log.data}\``,
        { parse_mode: 'Markdown' }
      );

      bot.answerCallbackQuery(query.id, { text: '✅ Product resent!' });
    } catch (err) {
      console.error(err);
      bot.answerCallbackQuery(query.id, { text: '❌ Error resending.', show_alert: true });
    }
  }
});

// 🆘 HELP command
bot.onText(/\/help/, (msg) => {
  const chatId = msg.chat.id;

  const message =
    `🆘 *Help Guide*\n\n` +
    `Use the following commands:\n` +
    `• /products — View available products\n` +
    `• /buy [id] — Buy a product\n` +
    `• /myorders — View your order history\n\n` +
    `Need support? Use /contact`;

  bot.sendMessage(chatId, message, { parse_mode: 'Markdown' });
});

// 📬 CONTACT command
bot.onText(/\/contact/, (msg) => {
  const chatId = msg.chat.id;

  const message =
    `📬 *Contact Support*\n\n` +
    `If you have any issues with your purchase, you can contact us via email:\n\n` +
    `✉️ support@dnfsolutions.net\n\n` +
    `Please include your Telegram username and product name for faster assistance.`;

  bot.sendMessage(chatId, message, { parse_mode: 'Markdown' });
});

module.exports = bot;
