const express = require('express');
const router = express.Router();
const { Product, PurchaseLog, ProductStock } = require('../models');
const { Op } = require('sequelize');
const { Parser } = require('json2csv');
const multer = require('multer');
const upload = multer({ dest: 'tmp/' });
const fs = require('fs');
const csv = require('csv-parser');
const { ErrorLog } = require('../models');

function ensureLoggedIn(req, res, next) {
  if (req.session && req.session.loggedIn) {
    return next();
  }
  res.redirect('/admin/login');
}

router.get('/login', (req, res) => {
  res.render('adminLogin', { error: null });
});

router.post('/login', (req, res) => {
  const { username, password } = req.body;
  if (username === process.env.ADMIN_USER && password === process.env.ADMIN_PASS) {
    req.session.loggedIn = true;
    res.redirect('/admin');
  } else {
    res.render('adminLogin', { error: 'Invalid credentials' });
  }
});

router.get('/logout', (req, res) => {
  req.session.destroy(() => {
    res.redirect('/admin/login');
  });
});

// GET /admin — Panel principal
router.get('/', ensureLoggedIn, async (req, res) => {
  const { q, type, from, to, inventory } = req.query;

  const productFilter = {};
  const logFilter = {};

  if (q) productFilter.name = { [Op.like]: `%${q}%` };
  if (type) {
    productFilter.type = type;
    logFilter.type = type;
  }
  if (from && to) {
    logFilter.createdAt = {
      [Op.between]: [new Date(from), new Date(to)]
    };
  }

  // 🧠 Filtro por tipo de inventario
  if (inventory === 'limited') {
    productFilter.is_unlimited = false;
  } else if (inventory === 'unlimited') {
    productFilter.is_unlimited = true;
  }

  const products = await Product.findAll({
    where: productFilter,
    order: [['id', 'DESC']],
    raw: true
  });

  const logs = await PurchaseLog.findAll({
    where: logFilter,
    order: [['createdAt', 'DESC']],
    limit: 100
  });

  const totalSales = logs.reduce((sum, l) => sum + parseFloat(l.price), 0);

  // Obtener stock por producto
  const { ProductStock } = require('../models');
  const stockMap = {};

  for (const product of products) {
    if (product.is_unlimited) {
      stockMap[product.id] = { total: null, available: '∞' };
    } else {
      const total = await ProductStock.count({ where: { product_id: product.id } });
      const available = await ProductStock.count({ where: { product_id: product.id, is_used: false } });
      stockMap[product.id] = { total, available };
    }
  }

  res.render('admin', {
    products,
    logs,
    filters: { q, type, from, to, inventory },
    totalSales,
    totalCount: logs.length,
    stockData: stockMap
  });
});

// POST /admin/add — Agregar nuevo producto
router.post('/add', ensureLoggedIn, async (req, res) => {
  const { name, type, data, price, is_unlimited } = req.body;

  try {
    await Product.create({
      name,
      type,
      data,
      price,
      is_unlimited: is_unlimited === 'on'
    });

    res.redirect('/admin');
      } catch (err) {
      console.error(err);
      await ErrorLog.create({
        context: 'Admin Add Product',
        message: err.message,
        stack: err.stack
      });
      res.send('❌ Error adding product');
    }

});

router.get('/error-logs', ensureLoggedIn, async (req, res) => {
  try {
    const logs = await ErrorLog.findAll({
      order: [['createdAt', 'DESC']],
      limit: 100
    });

    res.render('errorLogs', { logs });
  } catch (err) {
    console.error(err);
    res.status(500).send('❌ Could not load error logs');
  }
});

// Export to CSV
router.get('/export', ensureLoggedIn, async (req, res) => {
  const { type, from, to } = req.query;
  const filter = {};

  if (type) filter.type = type;
  if (from && to) {
    filter.createdAt = {
      [Op.between]: [new Date(from), new Date(to)]
    };
  }

  try {
    const logs = await PurchaseLog.findAll({ where: filter });

    const fields = ['id', 'telegram_id', 'name', 'type', 'price', 'createdAt'];
    const parser = new Parser({ fields });
    const csv = parser.parse(logs.map(l => l.toJSON()));

    res.header('Content-Type', 'text/csv');
    res.attachment('sales_log.csv');
    res.send(csv);
  } catch (err) {
    console.error(err);
    res.status(500).send('❌ Error exporting CSV');
  }
});

//Export Products
router.get('/export-products', ensureLoggedIn, async (req, res) => {
  try {
    const products = await Product.findAll();

    const fields = ['id', 'name', 'type', 'data', 'price', 'is_unlimited', 'is_used', 'is_reserved'];
    const parser = new Parser({ fields });
    const csv = parser.parse(products.map(p => p.toJSON()));

    res.header('Content-Type', 'text/csv');
    res.attachment('products.csv');
    res.send(csv);
  } catch (err) {
    console.error(err);
    res.status(500).send('❌ Error exporting products');
  }
});

//Import Products
router.post('/import-products', upload.single('file'), ensureLoggedIn, async (req, res) => {
  const results = [];

  try {
    fs.createReadStream(req.file.path)
      .pipe(csv())
      .on('data', (data) => results.push(data))
      .on('end', async () => {
        for (const item of results) {
          await Product.create({
            name: item.name,
            type: item.type,
            data: item.data,
            price: parseFloat(item.price),
            is_unlimited: item.is_unlimited === 'true'
          });
        }

        fs.unlinkSync(req.file.path); // eliminar archivo temporal
        res.redirect('/admin');
      });
  } catch (err) {
    console.error(err);
    res.status(500).send('❌ Error importing CSV');
  }
});

// Ver y agregar claves para un producto
router.get('/product-stock/:id', ensureLoggedIn, async (req, res) => {
  const productId = req.params.id;
  const show = req.query.show || 'all';

  const product = await Product.findByPk(productId);

  const where = { product_id: productId };
  if (show === 'used') {
    where.is_used = true;
  }

  const stock = await ProductStock.findAll({
    where,
    order: [['id', 'ASC']]
  });

  res.render('productStock', {
    product,
    stock,
    filter: show
  });
});

router.post('/product-stock/:id', ensureLoggedIn, async (req, res) => {
  const productId = req.params.id;
  const keys = req.body.keys?.split('\n').map(k => k.trim()).filter(Boolean) || [];

  await Promise.all(keys.map(key => ProductStock.create({
    product_id: productId,
    data: key
  })));

  res.redirect(`/admin/product-stock/${productId}`);
});

// Marcar clave como usada
router.post('/product-stock/:productId/mark-used/:stockId', ensureLoggedIn, async (req, res) => {
  const { productId, stockId } = req.params;
  await ProductStock.update({ is_used: true }, { where: { id: stockId, product_id: productId } });
  res.redirect(`/admin/product-stock/${productId}`);
});

// Eliminar clave
router.post('/product-stock/:productId/delete/:stockId', ensureLoggedIn, async (req, res) => {
  const { productId, stockId } = req.params;
  await ProductStock.destroy({ where: { id: stockId, product_id: productId } });
  res.redirect(`/admin/product-stock/${productId}`);
});

// Marcar como usada
router.post('/product-stock/:id/mark-used', ensureLoggedIn, async (req, res) => {
  const stockId = req.params.id;
  await ProductStock.update({ is_used: true }, { where: { id: stockId } });
  res.redirect('back');
});

// Eliminar clave
router.post('/product-stock/:id/delete', ensureLoggedIn, async (req, res) => {
  const stockId = req.params.id;
  await ProductStock.destroy({ where: { id: stockId } });
  res.redirect('back');
});

// Importar desde CSV
router.post('/product-stock/:id/import-csv', upload.single('file'), ensureLoggedIn, async (req, res) => {
  const productId = req.params.id;
  const results = [];

  try {
    fs.createReadStream(req.file.path)
      .pipe(csv())
      .on('data', (data) => results.push(data))
      .on('end', async () => {
        for (const item of results) {
          if (item.data) {
            await ProductStock.create({ product_id: productId, data: item.data.trim() });
          }
        }

        fs.unlinkSync(req.file.path);
        res.redirect(`/admin/product-stock/${productId}`);
      });
  } catch (err) {
    console.error(err);
    res.status(500).send('❌ Error importing keys CSV');
  }
});

// POST /admin/product-stock/:stockId/mark-used
router.post('/product-stock/:stockId/mark-used', ensureLoggedIn, async (req, res) => {
  const { stockId } = req.params;
  const stockItem = await ProductStock.findByPk(stockId);
  if (stockItem) {
    await stockItem.update({ is_used: true });
    res.redirect(`/admin/product-stock/${stockItem.product_id}`);
  } else {
    res.status(404).send('Key not found');
  }
});

// POST /admin/product-stock/:stockId/delete
router.post('/product-stock/:stockId/delete', ensureLoggedIn, async (req, res) => {
  const { stockId } = req.params;
  const stockItem = await ProductStock.findByPk(stockId);
  if (stockItem) {
    const productId = stockItem.product_id;
    await stockItem.destroy();
    res.redirect(`/admin/product-stock/${productId}`);
  } else {
    res.status(404).send('Key not found');
  }
});

router.post('/product-stock/:id/import-csv', upload.single('file'), ensureLoggedIn, async (req, res) => {
  const productId = req.params.id;
  const results = [];

  fs.createReadStream(req.file.path)
    .pipe(csvParser())
    .on('data', (row) => {
      if (row.data) {
        results.push({
          product_id: productId,
          data: row.data,
          is_used: row.is_used === 'true'
        });
      }
    })
    .on('end', async () => {
      await ProductStock.bulkCreate(results);
      fs.unlinkSync(req.file.path);
      res.redirect(`/admin/product-stock/${productId}`);
    });
});
module.exports = router;
