require('dotenv').config();
const Stripe = require("stripe");
const { Product, PurchaseLog, ProductStock } = require("./models");
const TelegramBot = require("node-telegram-bot-api");
const sendEmail = require("./utils/sendEmail");

const bot = new TelegramBot(process.env.TELEGRAM_BOT_TOKEN);

module.exports = async (req, res) => {
  console.log('📥 Webhook recibido');
  console.log('Headers:', req.headers);
  console.log('Probando body:', req.rawBody?.toString()?.slice(0, 300));

  const sig = req.headers["stripe-signature"];
  const rawBody = req.rawBody.toString();

  const stripe = new Stripe(process.env.STRIPE_SECRET_KEY);
  const endpointSecret = process.env.STRIPE_WEBHOOK_SECRET;

  try {
    const event = stripe.webhooks.constructEvent(rawBody, sig, endpointSecret);
    const eventId = event.id;

    console.log('✅ Evento Stripe recibido:', event.type);
    console.log('🧾 Metadata:', event?.data?.object?.metadata);

    const duplicate = await PurchaseLog.findOne({ where: { stripe_event_id: eventId } });
    if (duplicate) {
      console.warn('⚠️ Webhook duplicado ignorado:', eventId);
      return res.status(200).send('✅ Duplicate webhook ignored');
    }

    if (event.type === "checkout.session.completed") {
      const session = event.data.object;
      const telegramId = session.metadata.chat_id;
      const productId = session.metadata.product_id;
      const customerEmail = session.customer_details.email;

      const product = await Product.findOne({ where: { id: productId } });

      if (!product) {
        console.warn("❌ Producto no encontrado.");
        return res.status(400).send("Producto no encontrado");
      }

      let keyToDeliver = product.data;

      // 🔐 Entrega desde ProductStock si no es ilimitado
      if (!product.is_unlimited) {
        const stock = await ProductStock.findOne({
          where: {
            product_id: product.id,
            is_used: false
          },
          order: [['id', 'ASC']]
        });

        if (!stock) {
          console.warn("⚠️ No quedan claves disponibles para este producto.");
          await bot.sendMessage(telegramId, "⚠️ Sorry, this product is currently out of stock.");
          return res.status(200).send("Sin stock disponible");
        }

        keyToDeliver = stock.data;
        await stock.update({ is_used: true });

        await product.update({ is_reserved: false });
      }

      // ✅ Enviar por Telegram
      await bot.sendMessage(
        telegramId,
        `✅ *Thank you for your purchase!*\n\n` +
        `🛍️ *Product:* ${product.name}\n📦 *Type:* ${product.type}\n\n` +
        `🔑 *Key:* \`${keyToDeliver}\``,
        { parse_mode: "Markdown" }
      );

      // 🧾 Registrar en PurchaseLog
      await PurchaseLog.create({
        telegram_id: telegramId,
        product_id: product.id,
        name: product.name,
        type: product.type,
        data: keyToDeliver,
        price: product.price,
        stripe_event_id: eventId
      });

      // 📧 Enviar por correo
      if (customerEmail) {
        await sendEmail({
          to: customerEmail,
          subject: `🎉 Thank you for your purchase at DNF Solutions`,
          text: `Thank you for your purchase!\n\nProduct: ${product.name}\nType: ${product.type}\n\nKey:\n${keyToDeliver}\n\nIf you have any issues, contact us at support@dnfsolutions.net`,
          html: `
            <div style="font-family: Arial, sans-serif; padding: 20px; color: #333;">
              <h2 style="color: #1a73e8;">Thank you for your purchase!</h2>
              <p><strong>Product:</strong> ${product.name}</p>
              <p><strong>Type:</strong> ${product.type}</p>
              <p><strong>Key / Access Data:</strong></p>
              <pre style="background: #f4f4f4; padding: 10px; border-radius: 5px;">${keyToDeliver}</pre>
              <hr style="margin: 20px 0;">
              <p>If you have any questions or issues, feel free to reply to this email.</p>
              <p>—<br><strong>DNF Solutions</strong><br>
              support@dnfsolutions.net<br>
              Miami, FL, USA</p>
            </div>
          `
        });
      }

      console.log(`📦 Producto entregado al usuario ${telegramId}`);
    }

    res.status(200).send("✅ Webhook received");
  } catch (err) {
    console.error("❌ Webhook verification failed or processing error:", err.message);
    return res.status(400).send(`Webhook Error: ${err.message}`);
  }
};
